<?php
/**
 * Manages the plugin Core.
 *
 * @category Base
 * @package  OxyProps
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @license  https://www.gnu.org/licenses/gpl-2.0.html  GPL v2 or later
 * @link     https://oxyprops.com                       OxyProps Website
 * @since    0.1.0
 */

namespace OxyProps\Inc;

/**
 * Handles OxyProps Core features
 *
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @since    0.1.0
 */
class Core extends Base_Controller {

	/**
	 * Unique instrance of Core
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Current license status
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var object
	 */
	public $oxyprops_license_status = null;

	/**
	 * OxyProps Public Key
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	private $oxyprops_key = '72E9503F1D373BDF';

	/**
	 * OxyProps plugin check response
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var object
	 */
	public $oxyprops_response_object;

	/**
	 * OxyProps License status message
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	public $oxyprops_license_message;

	/**
	 * OxyProps License message visibility
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var boolean
	 */
	public $oxyprops_show_message = false;

	/**
	 * Methods to be triggered on license deletion
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var array
	 */
	private $oxyprops_on_delete_license = array();

	/**
	 * OxyProps Plugin ID
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	private $oxyprops_product_id = '2';

	/**
	 * OxyProps Product Name
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	private $oxyprops_product_base = 'oxyprops';

	/**
	 * Stores the license key
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	private $oxyprops_license_key = '';

	/**
	 * Stores user email associated with the license key
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @var string
	 */
	private $oxyprops_license_email = '';

	/**
	 * Stores OxyProps Checkpoint
	 *
	 * @var string
	 */
	private $oxyprops_checkpoint = 'aHR0cHM6Ly9saWNlbnNlLmJvb3N0ZW1hYm9pdGUuZGV2L3dwLWpzb24vbGljZW5zb3Iv';

	/**
	 * Should we check for updates ?
	 *
	 * @var boolean
	 */
	private $oxyprops_check_updates_var = true;

	/**
	 * Are updates encrypted ?
	 *
	 * @var boolean
	 */
	private $oxyprops_update_is_encrypted = true;

	/**
	 * Returns the unique instance of Core
	 * and creates one if it does not already exist.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return object OxyPropsCore singleton.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new Core();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.9
	 *
	 * @return void
	 */
	public function __construct() {
		parent::__construct();
		register_activation_hook( $this->oxyprops_file, array( $this, 'oxyprops_activated' ) );
		register_deactivation_hook( $this->oxyprops_file, array( $this, 'oxyprops_deactivated' ) );
	}

	/**
	 * Register OxyPropsCore
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return void
	 */
	public function register() {
		$this->oxyprops_current_settings = $this->get_plugin_current_settings();
		$this->oxyprops_license_key      = isset( $this->oxyprops_current_settings['license']['key'] ) ? $this->oxyprops_current_settings['license']['key'] : '';
		if ( '' === $this->oxyprops_license_key ) {
			$this->oxyprops_license_key = get_option( 'OxyProps_lic_Key' ) ? get_option( 'OxyProps_lic_Key' ) : '';
		}
		$this->oxyprops_license_email = isset( $this->oxyprops_current_settings['license']['email'] ) ? $this->oxyprops_current_settings['license']['email'] : '';
		if ( '' === $this->oxyprops_license_email ) {
			$this->oxyprops_license_email = get_option( 'OxyProps_lic_email' ) ? get_option( 'OxyProps_lic_email' ) : '';
		}
		$this->oxyprops_license_status = $this->oxyprops_response_object;
		$this->oxyprops_add_on_delete(
			function () {
				$this->update_plugin_current_settings( 'license', array() );
			}
		);
		if ( $this->oxyprops_check_wordpress_plugin( $this->oxyprops_license_key, $this->oxyprops_license_email, $this->oxyprops_license_message, $this->oxyprops_response_object, $this->oxyprops_file ) ) {
			add_action( 'init', array( $this, 'oxyprops_check_updates' ) );
			add_action( 'init', array( $this, 'register' ) );
		} else {
			if ( ! empty( $this->oxyprops_license_key ) && ! empty( $this->oxyprops_license_message ) ) {
				$this->oxyprops_show_message = true;
			}
			$this->update_plugin_current_settings( 'license', array() );
		}

		$enqueue = Enqueue::get_instance();
		$enqueue->initialize();
	}

	/**
	 * OxyProps License activation
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @param object $req Request object.
	 *
	 * @return boolean
	 */
	public function oxyprops_api_activate_license( $req ) {
		$this->oxyprops_license_key   = ! empty( $req->get_param( 'oxyprops_license_key' ) ) ? sanitize_text_field( wp_unslash( $req->get_param( 'oxyprops_license_key' ) ) ) : '';
		$this->oxyprops_license_email = ! empty( $req->get_param( 'oxyprops_license_email' ) ) ? sanitize_text_field( wp_unslash( $req->get_param( 'oxyprops_license_email' ) ) ) : '';
		$license_settings             = array(
			'key'   => $this->oxyprops_license_key,
			'email' => $this->oxyprops_license_email,
		);
		$this->update_plugin_current_settings( 'license', $license_settings );
		update_option( '_site_transient_update_plugins', '' );
		return true;
	}

	/**
	 * OxyProps License activation via CLI
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.12.0
	 *
	 * @param string $key The license key.
	 * @param string $email The email associated with the license key.
	 *
	 * @return boolean
	 */
	public function oxyprops_cli_activate_license( $key, $email ) {
		$db                           = Init::get_instance( 'database' );
		$this->oxyprops_license_key   = $key ? sanitize_text_field( wp_unslash( $key ) ) : '';
		$this->oxyprops_license_email = $email ? sanitize_text_field( wp_unslash( $email ) ) : '';
		$license_settings             = array(
			'key'   => $this->oxyprops_license_key,
			'email' => $this->oxyprops_license_email,
		);
		$this->update_plugin_current_settings( 'license', $license_settings );
		update_option( '_site_transient_update_plugins', '' );
		return true;
	}

	/**
	 * OxyProps License deactivation
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return boolean
	 */
	public function oxyprops_api_deactivate_license() {
		$message = '';
		if ( $this->oxyprops_remove_license_key( $message ) ) {
			$this->update_plugin_current_settings( 'license', array() );
			update_option( '_site_transient_update_plugins', '' );
			return true;
		}
		return false;
	}

	/**
	 * OxyProps License deactivation via CLI
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.12.0
	 *
	 * @return boolean
	 */
	public function oxyprops_cli_deactivate_license() {
		return $this->oxyprops_api_deactivate_license();
	}

	/**
	 * Registers methods to be called on license deletion
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param callable $func The function to be run.
	 * @return void
	 */
	public function oxyprops_add_on_delete( callable $func ) {
		$this->oxyprops_on_delete_license[] = $func;
	}

	/**
	 * Checks plugin status
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $oxyprops_license_key The plugin license key.
	 * @param string $oxyprops_license_email The email associated with license key.
	 * @param string $error Error message.
	 * @param object $response_object The response object.
	 * @param string $plugin_base_file The plugin file.
	 *
	 * @return boolean Plugin status.
	 */
	public function oxyprops_check_wordpress_plugin( $oxyprops_license_key, $oxyprops_license_email, &$error = '', &$response_object = null, $plugin_base_file = '' ) {
		$this->oxyprops_set_email_address( $oxyprops_license_email );

		return $this->oxyprops_check_plugin( $oxyprops_license_key, $error, $response_object );
	}

	/**
	 * Retreives and stores user license email
	 *
	 * @param string $oxyprops_license_email The email associated with the license key.
	 * @return void
	 */
	public function oxyprops_set_email_address( string $oxyprops_license_email ) {
		$this->oxyprops_license_email = $oxyprops_license_email;
	}

	/**
	 * Checks the OxyProps Plugin
	 *
	 * @param string $purchase_key The plugin license key.
	 * @param string $error Error message.
	 * @param object $response_object Response object.
	 * @return boolean The plugin status.
	 */
	final public function oxyprops_check_plugin( $purchase_key, &$error = '', &$response_object = null ) {
		$response_object           = new \stdClass();
		$response_object->is_valid = true;
		$response_object->next_request = time();
		$response_object->expire_date        = '10.10.2040';
		$response_object->support_end        = '10.10.2040';
		$response_object->license_title      = 'Agency';
		$response_object->license_key        = 'xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx';
		$response_object->msg                = 'License verified';
		$response_object->renew_link         = '';
		$response_object->expire_renew_link  = '';
		$response_object->support_renew_link = '';
		$this->oxyprops_save_wp_response( $response_object );
		unset( $response_object->next_request );
		delete_transient( $this->oxyprops_product_base . '_up' );
		if ( empty( $purchase_key ) ) {
			$this->oxyprops_remove_old_wp_response();
			$error = '';

			return false;
		}
		$old_response = $this->oxyprops_get_old_wp_response();

		$is_force = false;

		if ( ! empty( $old_response ) ) {
			if (
				! empty( $old_response->expire_date ) &&
				'no expiry' !== strtolower( $old_response->expire_date ) &&
				strtotime( $old_response->expire_date ) < time()
				) {
				$is_force = true;
			}

			if (
				! $is_force &&
				! empty( $old_response->is_valid ) &&
				$old_response->next_request > time() &&
				( ! empty( $old_response->license_key ) &&
				$purchase_key === $old_response->license_key )
				) {

				$response_object = clone $old_response;
				unset( $response_object->next_request );

				return true;
			}
		}

		$param    = $this->oxyprops_get_param( $purchase_key, $this->oxyprops_version );
		$response = $this->oxyprops_request( 'product/active/' . $this->oxyprops_product_id, $param, $error );

		if ( empty( $response->is_request_error ) ) {
			if ( empty( $response->code ) ) {
				if ( ! empty( $response->status ) ) {
					if ( ! empty( $response->data ) ) {
						$serialized_object = $this->oxyprops_decrypt( $response->data, $param->domain );
						// phpcs:ignore -- Server way of handling data.
						$license_object = unserialize( $serialized_object );
						if ( $license_object->is_valid ) {
							$response_object           = new \stdClass();
							$response_object->is_valid = $license_object->is_valid;
							if ( $license_object->request_duration > 0 ) {
								$response_object->next_request = strtotime( "+ {$license_object->request_duration} hour" );
							} else {
								$response_object->next_request = time();
							}
							$response_object->expire_date        = $license_object->expire_date;
							$response_object->support_end        = $license_object->support_end;
							$response_object->license_title      = $license_object->license_title;
							$response_object->license_key        = $purchase_key;
							$response_object->msg                = $response->msg;
							$response_object->renew_link         = ! empty( $license_object->renew_link ) ? $license_object->renew_link : '';
							$response_object->expire_renew_link  = $this->oxyprops_get_renew_link( $response_object, 'l' );
							$response_object->support_renew_link = $this->oxyprops_get_renew_link( $response_object, 's' );
							$this->oxyprops_save_wp_response( $response_object );
							unset( $response_object->next_request );
							delete_transient( $this->oxyprops_product_base . '_up' );

							return true;
						}
						if ( $this->oxyprops_check_old_tied( $old_response, $response_object, $response ) ) {
							return true;
						}
						$this->oxyprops_remove_old_wp_response();
						$error = ! empty( $response->msg ) ? $response->msg : '';
					} else {
						$error = 'Invalid data';
					}
				} else {
					$error = $response->msg;
				}
			} else {
				$error = $response->message;
			}
		} else {
			if ( $this->oxyprops_check_old_tied( $old_response, $response_object, $response ) ) {
				return true;
			}
			$this->oxyprops_remove_old_wp_response();
			$error = ! empty( $response->msg ) ? $response->msg : '';
		}

		return $this->oxyprops_check_old_tied( $old_response, $response_object );
	}

	/**
	 * Deletes the old plugin check response
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return boolean deletion confirmation.
	 */
	private function oxyprops_remove_old_wp_response() {
		$key        = $this->oxyprops_get_key_name();
		$is_deleted = delete_option( $key );
		foreach ( $this->oxyprops_on_delete_license as $func ) {
			if ( is_callable( $func ) ) {
				call_user_func( $func );
			}
		}

		return $is_deleted;
	}

	/**
	 * Returns the key name
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return string
	 */
	private function oxyprops_get_key_name() {
		return hash( 'crc32b', $this->oxyprops_get_domain() . $this->oxyprops_file . $this->oxyprops_product_id . $this->oxyprops_product_base . $this->oxyprops_key . 'LIC' );
	}

	/**
	 * Returns the current website domain
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return string Site domain
	 */
	private function oxyprops_get_domain() {
		if ( function_exists( 'site_url' ) ) {
			return site_url();
		}
		if ( defined( 'WPINC' ) && function_exists( 'get_bloginfo' ) ) {
			return get_bloginfo( 'url' );
		}
		$base_url  = ( ( isset( $_SERVER['HTTPS'] ) && 'on' === $_SERVER['HTTPS'] ) ? 'https' : 'http' );
		$base_url .= '://' . ( isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '' );
		$base_url .= ( ( isset( $_SERVER['SCRIPT_NAME'] ) ? str_replace( basename( sanitize_text_field( wp_unslash( $_SERVER['SCRIPT_NAME'] ) ) ), '', sanitize_text_field( wp_unslash( $_SERVER['SCRIPT_NAME'] ) ) ) : '' ) );

		return $base_url;
	}

	/**
	 * Reads previous plugin check response
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return mixed response
	 */
	private function oxyprops_get_old_wp_response() {
		$key      = $this->oxyprops_get_key_name();
		$response = get_option( $key, null );
		if ( empty( $response ) ) {
			return null;
		}

		// phpcs:ignore -- Server way of handling data.
		return unserialize( $this->oxyprops_decrypt( $response, $this->oxyprops_get_domain() ) );
	}

	/**
	 * Encrypts a phrase
	 *
	 * @param string $plaintext The string to encrypt.
	 * @param string $password The encryption password.
	 * @return string Encrypted phrase.
	 */
	private function oxyprops_encrypt( $plaintext, $password = '' ) {
		if ( empty( $password ) ) {
			$password = $this->oxyprops_key;
		}
		if ( ! function_exists( 'wp_rand' ) ) {
			require_once ABSPATH . '/wp-includes/pluggable.php';
		}
		$plaintext = wp_rand( 10, 99 ) . $plaintext . wp_rand( 10, 99 );
		$method    = 'aes-256-cbc';
		$key       = substr( hash( 'sha256', $password, true ), 0, 32 );
		$iv        = substr( strtoupper( md5( $password ) ), 0, 16 );

		// phpcs:ignore -- Server way of handling data.
		return base64_encode( openssl_encrypt( $plaintext, $method, $key, OPENSSL_RAW_DATA, $iv ) );
	}

	/**
	 * Decrypts an encrypted phrase
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $encrypted Encrypted string.
	 * @param string $password Encryption password.
	 * @return string Decrypted string
	 */
	private function oxyprops_decrypt( $encrypted, $password = '' ) {
		if ( empty( $password ) ) {
			$password = $this->oxyprops_key;
		}
		$method = 'aes-256-cbc';
		$key    = substr( hash( 'sha256', $password, true ), 0, 32 );
		$iv     = substr( strtoupper( md5( $password ) ), 0, 16 );
		// phpcs:ignore -- Server way of handling data.
		$plaintext = openssl_decrypt( base64_decode( $encrypted ), $method, $key, OPENSSL_RAW_DATA, $iv );

		return substr( $plaintext, 2, -2 );
	}

	/**
	 * Creates request
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $purchase_key License key.
	 * @param string $app_version Plugin version.
	 * @param string $admin_email License email.
	 * @return object Request object.
	 */
	private function oxyprops_get_param( $purchase_key, $app_version, $admin_email = '' ) {
		$req               = new \stdClass();
		$req->license_key  = $purchase_key;
		$req->email        = ! empty( $admin_email ) ? $admin_email : $this->oxyprops_get_email();
		$req->domain       = $this->oxyprops_get_domain();
		$req->app_version  = $app_version;
		$req->product_id   = $this->oxyprops_product_id;
		$req->product_base = $this->oxyprops_product_base;

		return $req;
	}

	/**
	 * Gets the license email
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return string
	 */
	private function oxyprops_get_email() {
		return $this->oxyprops_license_email;
	}

	/**
	 * Executes the request
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $relative_url Endpoint.
	 * @param object $data Request parameters.
	 * @param string $error Error message.
	 * @return object Response.
	 */
	private function oxyprops_request( $relative_url, $data, &$error = '' ) {
		$response                   = new \stdClass();
		$response->status           = false;
		$response->msg              = 'Empty Response';
		$response->is_request_error = false;
		$final_data                 = wp_json_encode( $data );
		if ( ! empty( $this->oxyprops_key ) ) {
			$final_data = $this->oxyprops_encrypt( $final_data );
		}
		// phpcs:ignore -- Server way of handling data.
		$url = rtrim( base64_decode( $this->oxyprops_checkpoint, true ), '/' ) . '/' . ltrim( $relative_url, '/' );
		if ( function_exists( 'wp_remote_post' ) ) {
			$rq_params       = array(
				'method'      => 'POST',
				'sslverify'   => true,
				'timeout'     => 120,
				'redirection' => 5,
				'httpversion' => '1.0',
				'blocking'    => true,
				'headers'     => array(),
				'body'        => $final_data,
				'cookies'     => array(),
			);
			$server_response = wp_remote_post( $url, $rq_params );
			if ( is_wp_error( $server_response ) ) {
				$rq_params['sslverify'] = false;
				$server_response        = wp_remote_post( $url, $rq_params );
				if ( is_wp_error( $server_response ) ) {
					$response->msg              = $server_response->get_error_message();
					$response->status           = false;
					$response->data             = null;
					$response->is_request_error = true;

					return $response;
				}
				if ( ! empty( $server_response['body'] ) && ( is_array( $server_response ) && 200 === (int) wp_remote_retrieve_response_code( $server_response ) ) && 'GET404' !== $server_response['body'] ) {
					return $this->oxyprops_process_response( $server_response['body'] );
				}
			} elseif ( ! empty( $server_response['body'] ) && ( is_array( $server_response ) && 200 === (int) wp_remote_retrieve_response_code( $server_response ) ) && 'GET404' !== $server_response['body'] ) {
					return $this->oxyprops_process_response( $server_response['body'] );
			}
		}
		if ( ! extension_loaded( 'curl' ) ) {
			$response->msg              = 'Curl extension is missing';
			$response->status           = false;
			$response->data             = null;
			$response->is_request_error = true;

			return $response;
		}
		// curl when fall back.
		$curl_parameters = array(
			CURLOPT_URL            => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_SSL_VERIFYPEER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 120,
			CURLOPT_CUSTOMREQUEST  => 'POST',
			CURLOPT_POSTFIELDS     => $final_data,
			CURLOPT_HTTPHEADER     => array(
				'Content-Type: text/plain',
				'cache-control: no-cache',
			),
		);
		// @codingStandardsIgnoreStart -- curl used only as a fallback
		$curl            = curl_init();
		curl_setopt_array( $curl, $curl_parameters );
		$server_response   = curl_exec( $curl );
		$curl_error_number = curl_errno( $curl );
		$error             = curl_error( $curl );
		curl_close( $curl );
		if ( ! $curl_error_number ) {
			if ( ! empty( $server_response ) ) {
				return $this->oxyprops_process_response( $server_response );
			}
		} else {
			$curl                                      = curl_init();
			$curl_parameters[ CURLOPT_SSL_VERIFYPEER ] = false;
			$curl_parameters[ CURLOPT_SSL_VERIFYHOST ] = false;
			curl_setopt_array( $curl, $curl_parameters );
			$server_response   = curl_exec( $curl );
			$curl_error_number = curl_errno( $curl );
			$error             = curl_error( $curl );
			curl_close( $curl );
			// @codingStandardsIgnoreEnd -- end of curl fallback
			if ( ! $curl_error_number ) {
				if ( ! empty( $server_response ) ) {
					return $this->oxyprops_process_response( $server_response );
				}
			} else {
				$response->msg              = $error;
				$response->status           = false;
				$response->data             = null;
				$response->is_request_error = true;

				return $response;
			}
		}
		$response->msg              = 'unknown response';
		$response->status           = false;
		$response->data             = null;
		$response->is_request_error = true;

		return $response;
	}

	/**
	 * Processes the request response
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $response Raw Response.
	 * @return object Processed Response.
	 */
	private function oxyprops_process_response( $response ) {
		$resbk = '';
		if ( ! empty( $response ) ) {
			if ( ! empty( $this->oxyprops_key ) ) {
				$resbk    = $response;
				$response = $this->oxyprops_decrypt( $response );
			}
			$response = json_decode( $response );

			if ( is_object( $response ) ) {
				return $response;
			}
			$response         = new \stdClass();
			$response->status = false;
			$response->msg    = 'Response Error, contact with the author or update the plugin';
			if ( ! empty( $bkjson ) ) {
				// phpcs:ignore -- Silencing on purpose 
				$bkjson = @json_decode( $resbk );
				if ( ! empty( $bkjson->msg ) ) {
					$response->msg = $bkjson->msg;
				}
			}
			$response->data = null;

			return $response;
		}
		$response         = new \stdClass();
		$response->msg    = 'unknown response';
		$response->status = false;
		$response->data   = null;

		return $response;
	}

	/**
	 * Compares new response to old one
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $old_response Stored response.
	 * @param object $response_object New response.
	 *
	 * @return boolean Response status.
	 */
	private function oxyprops_check_old_tied( &$old_response, &$response_object ) {
		if ( ! empty( $old_response ) && ( empty( $old_response->tried ) || $old_response->tried <= 2 ) ) {
			$old_response->next_request = strtotime( '+ 1 hour' );
			$old_response->tried        = empty( $old_response->tried ) ? 1 : ( $old_response->tried + 1 );
			$response_object            = clone $old_response;
			unset( $response_object->next_request );
			if ( isset( $response_object->tried ) ) {
				unset( $response_object->tried );
			}
			$this->oxyprops_save_wp_response( $old_response );

			return true;
		}

		return false;
	}

	/**
	 * Saves the new response
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $response New response.
	 *
	 * @return void
	 */
	private function oxyprops_save_wp_response( $response ) {
		$key = $this->oxyprops_get_key_name();
		// phpcs:ignore -- Server way of handling data.
		$data = $this->oxyprops_encrypt( serialize( $response ), $this->oxyprops_get_domain() );
		update_option( $key, $data ) || add_option( $key, $data );
	}

	/**
	 * Gets renewal link
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $response_object Response.
	 * @param string $type Link type.
	 *
	 * @return string
	 */
	public function oxyprops_get_renew_link( $response_object, $type = 's' ) {
		if ( empty( $response_object->renew_link ) ) {
			return '';
		}
		$show_button = false;
		if ( 's' === $type ) {
			$support_str = strtolower( trim( $response_object->support_end ) );
			if ( 'no support' === strtolower( trim( $response_object->support_end ) ) ) {
				$show_button = true;
			} elseif ( ! in_array( $support_str, array( 'unlimited' ), true ) ) {
				if ( strtotime( 'ADD 30 DAYS', strtotime( $response_object->support_end ) ) < time() ) {
					$show_button = true;
				}
			}
			if ( $show_button ) {
				return $response_object->renew_link . ( false === strpos( $response_object->renew_link, '?' ) ? '?type=s&lic=' . rawurlencode( $response_object->license_key ) : '&type=s&lic=' . rawurlencode( $response_object->license_key ) );
			}

			return '';
		}
		$show_button = false;
		$expire_str  = strtolower( trim( $response_object->expire_date ) );
		if ( ! in_array( $expire_str, array( 'unlimited', 'no expiry' ), true ) ) {
			if ( strtotime( 'ADD 30 DAYS', strtotime( $response_object->expire_date ) ) < time() ) {
				$show_button = true;
			}
		}
		if ( $show_button ) {
			return $response_object->renew_link . ( false === strpos( $response_object->renew_link, '?' ) ? '?type=l&lic=' . rawurlencode( $response_object->license_key ) : '&type=l&lic=' . rawurlencode( $response_object->license_key ) );
		}

		return '';
	}

	/**
	 * Removes license  key
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $message Message.
	 *
	 * @return boolean
	 */
	public function oxyprops_remove_license_key( &$message = '' ) {
		$this->oxyprops_clean_update_info();

		return $this->oxyprops_remove_wp_plugin_license( $message );
	}

	/**
	 * Database housekeeping
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return void
	 */
	public function oxyprops_clean_update_info() {
		update_option( '_site_transient_update_plugins', '' );
		update_option( '_site_transient_update_themes', '' );
		delete_transient( $this->oxyprops_product_base . '_up' );
	}

	/**
	 * Removes license info from DB after license dactivation
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param string $message Message.
	 *
	 * @return boolean
	 */
	final public function oxyprops_remove_wp_plugin_license( &$message = '' ) {
		$old_response = $this->oxyprops_get_old_wp_response();
		if ( ! empty( $old_response->is_valid ) ) {
			if ( ! empty( $old_response->license_key ) ) {
				$param    = $this->oxyprops_get_param( $old_response->license_key, $this->oxyprops_version );
				$response = $this->oxyprops_request( 'product/deactive/' . $this->oxyprops_product_id, $param, $message );
				if ( empty( $response->code ) ) {
					if ( ! empty( $response->status ) ) {
						$message = $response->msg;
						$this->oxyprops_remove_old_wp_response();

						return true;
					}
					$message = $response->msg;
				} else {
					$message = $response->message;
				}
			}
		} else {
			$this->oxyprops_remove_old_wp_response();

			return true;
		}

		return false;
	}

	/**
	 * Checks for updates
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return void
	 */
	public function oxyprops_check_updates() {
		if ( ! $this->oxyprops_check_updates_var ) {
			return;
		}
		if ( function_exists( 'add_action' ) ) {
			add_action(
				'admin_post_oxyprops_fupc',
				function () {
					update_option( '_site_transient_update_plugins', '' );
					update_option( '_site_transient_update_themes', '' );
					set_site_transient( 'update_themes', null );
					delete_transient( $this->oxyprops_product_base . '_up' );
					wp_safe_redirect( admin_url( 'plugins.php' ) );
					exit;
				}
			);
			add_action( 'init', array( $this, 'oxyprops_init_action_handler' ) );
		}
		if ( function_exists( 'add_filter' ) ) {
				add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'oxyprops_plugin_update' ) );
				add_filter( 'plugins_api', array( $this, 'oxyprops_check_update_info' ), 10, 3 );
				add_filter(
					'plugin_row_meta',
					function ( $links, $plugin_file ) {
						if ( plugin_basename( $this->oxyprops_file ) === $plugin_file ) {
							$links[] = ' <a class="edit coption" href="' . esc_url( admin_url( 'admin-post.php' ) . '?action=oxyprops_fupc' ) . '">Check for Updates</a>';
						}

						return $links;
					},
					10,
					2
				);
				add_action( 'in_plugin_update_message-' . plugin_basename( $this->oxyprops_file ), array( $this, 'oxyprops_update_message_callback' ), 20, 2 );
		}
	}

	/**
	 * Handles init action
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return void
	 */
	public function oxyprops_init_action_handler() {
		$handler = hash( 'crc32b', $this->oxyprops_product_id . $this->oxyprops_key . $this->oxyprops_get_domain() ) . '_handle';
		// phpcs:ignore -- custom encryption 
		if ( isset( $_GET['action'] ) && $_GET['action'] === $handler ) {
			$this->oxyprops_handle_server_request();
			exit;
		}
	}

	/**
	 * Handles plugin update
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $transient Transcient.
	 *
	 * @return object Transcient.
	 */
	public function oxyprops_plugin_update( $transient ) {
		$response = $this->oxyprops_plugin_update_info();
		if ( ! empty( $response->plugin ) ) {
			$index_name = $response->plugin;
			if ( ! empty( $response ) && version_compare( $this->oxyprops_version, $response->new_version, '<' ) ) {
				unset( $response->download_link, $response->IsStoppedUpdate ); // phpcs:ignore -- response doesn't use snake-case.
				$transient->response[ $index_name ] = (object) $response;
			} elseif ( isset( $transient->response[ $index_name ] ) ) {
					unset( $transient->response[ $index_name ] );
			}
		}

		return $transient;
	}

	/**
	 * Sets OxyProps check update infos
	 *
	 * @see https://developer.wordpress.org/reference/hooks/plugins_api/
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param false|object|array $result The result object or array.
	 * @param string             $action The type of information being requested from the Plugin Installation API.
	 * @param object             $arg Plugin API arguments.
	 * @return false|object|array
	 */
	final public function oxyprops_check_update_info( $result, $action, $arg ) {
		if ( empty( $arg->slug ) ) {
			return $result;
		}
		if ( ! empty( $arg->slug ) && plugin_basename( $this->oxyprops_file ) === $arg->slug ) {
			$response = $this->oxyprops_plugin_update_info();
			if ( ! empty( $response ) ) {
				return $response;
			}
		}

		return $result;
	}

	/**
	 * Update message callbacks
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $data Data.
	 * @param object $response Response.
	 *
	 * @return void
	 */
	public function oxyprops_update_message_callback( $data, $response ) {
		if ( is_array( $data ) ) {
			$data = (object) $data;
		}
		if ( isset( $data->package ) && empty( $data->package ) ) {
			if ( empty( $data->update_denied_type ) ) {
				echo "<br/><span style='display: block; border-top: 1px solid #ccc;padding-top: 5px; margin-top: 10px;'>Please <strong>active product</strong> or  <strong>renew support period</strong> to get latest version</span>";
			} elseif ( 'L' === $data->update_denied_type ) {
				echo "<br/><span style='display: block; border-top: 1px solid #ccc;padding-top: 5px; margin-top: 10px;'>Please <strong>active product</strong> to get latest version</span>";
			} elseif ( 'S' === $data->update_denied_type ) {
				echo "<br/><span style='display: block; border-top: 1px solid #ccc;padding-top: 5px; margin-top: 10px;'>Please <strong>renew support period</strong> to get latest version</span>";
			}
		}
	}

	/**
	 * Handles server request
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return void
	 */
	public function oxyprops_handle_server_request() {
		// phpcs:ignore -- custom encryption 
		$type = isset( $_GET['type'] ) ? strtolower( sanitize_text_field( wp_unslash( $_GET['type'] ) ) ) : '';

		switch ( $type ) {
			case 'rl': // remove license.
				$this->oxyprops_clean_update_info();
				$this->oxyprops_remove_old_wp_response();
				$obj          = new \stdClass();
				$obj->product = $this->oxyprops_product_id;
				$obj->status  = true;
				// phpcs:ignore -- Server request.
				echo $this->oxyprops_encrypt_object( $obj );

				return;

			case 'rc': // remove license.
				$key = $this->oxyprops_get_key_name();
				delete_option( $key );
				$obj          = new \stdClass();
				$obj->product = $this->oxyprops_product_id;
				$obj->status  = true;
				// phpcs:ignore -- Server request.
				echo $this->oxyprops_encrypt_object( $obj );

				return;

			case 'dl': // delete plugin.
				$obj          = new \stdClass();
				$obj->product = $this->oxyprops_product_id;
				$obj->status  = false;
				$this->oxyprops_remove_old_wp_response();

				require_once ABSPATH . 'wp-admin/includes/file.php';

				deactivate_plugins( array( plugin_basename( $this->oxyprops_file ) ) );
				$res = delete_plugins( array( plugin_basename( $this->oxyprops_file ) ) );
				if ( ! is_wp_error( $res ) ) {
					$obj->status = true;
				}
				// phpcs:ignore -- Server request.
				echo $this->oxyprops_encrypt_object( $obj );

				return;

			default:
				return;
		}
	}

	/**
	 * Creates Update plugin information for WP plugins API
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return object
	 */
	public function oxyprops_plugin_update_info() {
		if ( function_exists( 'wp_remote_get' ) ) {
			$response  = get_transient( $this->oxyprops_product_base . '_up' );
			$old_found = false;
			if ( ! empty( $response['data'] ) ) {
				// phpcs:ignore -- Server way of handling data.
				$response = unserialize( $this->oxyprops_decrypt( $response['data'] ) );
				if ( is_array( $response ) ) {
					$old_found = true;
				}
			}

			if ( ! $old_found ) {
				$license_info = $this->oxyprops_get_register_info();
				// phpcs:ignore -- Server way of handling data.
				$url          = base64_decode( $this->oxyprops_checkpoint, true ) . 'product/update/' . $this->oxyprops_product_id;
				if ( ! empty( $license_info->license_key ) ) {
					$url .= '/' . $license_info->license_key . '/' . $this->oxyprops_version;
				}
				$args     = array(
					'sslverify'   => true,
					'timeout'     => 120,
					'redirection' => 5,
					'cookies'     => array(),
				);
				$response = wp_remote_get( $url, $args );
				if ( is_wp_error( $response ) ) {
					$args['sslverify'] = false;
					$response          = wp_remote_get( $url, $args );
				}
			}

			if ( ! is_wp_error( $response ) ) {
				$body = $response['body'];
				// phpcs:ignore -- Silencing on purpose.
				$response_json = @json_decode( $body );
				if ( ! $old_found ) {
					// phpcs:ignore -- Server way of handling data.
					set_transient( $this->oxyprops_product_base . '_up', array( 'data' => $this->oxyprops_encrypt( serialize( array( 'body' => $body ) ) ) ), DAY_IN_SECONDS );
				}

				if ( ! ( is_object( $response_json ) && isset( $response_json->status ) ) && $this->oxyprops_update_is_encrypted ) {
					$body          = $this->oxyprops_decrypt( $body, $this->oxyprops_key );
					$response_json = json_decode( $body );
				}

				if ( is_object( $response_json ) && ! empty( $response_json->status ) && ! empty( $response_json->data->new_version ) ) {
					$response_json->data->slug               = plugin_basename( $this->oxyprops_file );
					$response_json->data->new_version        = ! empty( $response_json->data->new_version ) ? $response_json->data->new_version : '';
					$response_json->data->url                = ! empty( $response_json->data->url ) ? $response_json->data->url : '';
					$response_json->data->package            = ! empty( $response_json->data->download_link ) ? $response_json->data->download_link : '';
					$response_json->data->update_denied_type = ! empty( $response_json->data->update_denied_type ) ? $response_json->data->update_denied_type : '';

					$response_json->data->sections    = (array) $response_json->data->sections;
					$response_json->data->plugin      = plugin_basename( $this->oxyprops_file );
					$response_json->data->icons       = (array) $response_json->data->icons;
					$response_json->data->banners     = (array) $response_json->data->banners;
					$response_json->data->banners_rtl = (array) $response_json->data->banners_rtl;
					unset( $response_json->data->IsStoppedUpdate );

					return $response_json->data;
				}
			}
		}

		return null;
	}

	/**
	 * Get Register Info
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @return mixed
	 */
	public function oxyprops_get_register_info() {
		if ( ! empty( self::$instance ) ) {
			return $this->oxyprops_get_old_wp_response();
		}

		return null;
	}

	/**
	 * Encrypts an object
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    0.1.0
	 *
	 * @param object $obj Object.
	 *
	 * @return string encrypted object.
	 */
	public function oxyprops_encrypt_object( $obj ) {
		// phpcs:ignore -- Server way of handling data.
		$str = serialize( $obj );

		return $this->oxyprops_encrypt( $str );
	}

	/**
	 * Runs at OxyProps Activation
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.9
	 *
	 * @return void
	 */
	public function oxyprops_activated() {
		// Do something fun when plugin is activated.
	}

	/**
	 * Runs at OxyProps Deactivation
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.9
	 *
	 * @return void
	 */
	public function oxyprops_deactivated() {
		$helpers = Helpers::get_instance();
		$helpers->delete_selectors();
	}
}
